from django.shortcuts import render, redirect, get_object_or_404
from django.db.models import Q
from .models import Wave1Question, Wave2Question, EventDetails, ExplorationData
from .forms import ExplorationDataFilterForm
from django.http import JsonResponse, HttpResponseBadRequest, HttpResponseRedirect
from django.views.decorators.http import require_POST
from django.utils import timezone
from datetime import timedelta
from django.utils.safestring import mark_safe
from django.contrib.sessions.models import Session
import json
import re

CATEGORIES = [
    {'name': 'Plan', 'prefix': 'P'},
    {'name': 'Collect', 'prefix': 'C'},
    {'name': 'Analyze', 'prefix': 'A'},
    {'name': 'QA', 'prefix': 'Q'},
    {'name': 'Share', 'prefix': 'S'},
    {'name': 'Re-Use', 'prefix': 'R'},
]

def select_game_mode(request):
    # Check if there's an ongoing quiz
    ongoing_quiz = False
    current_category = request.session.get('current_category')
    current_index = request.session.get('current_index')

    if current_category is not None and current_index is not None:
        ongoing_quiz = True

    # Calculate all_categories_completed
    completed_categories = request.session.get('completed_categories', [])
    all_wave2_questions_answered = request.session.get('all_wave2_questions_answered', False)
    total_categories = len(CATEGORIES)
    all_categories_completed = len(completed_categories) == total_categories

    # Handle the POST request (choosing a game mode)
    if request.method == 'POST':
        selected_game_mode = request.POST.get('game_mode')
        if selected_game_mode == 'Quality Assurance Mode':
            request.session['selected_game_mode'] = selected_game_mode
            return redirect('select_category')
        elif selected_game_mode == 'Exploration Mode':
            request.session['selected_game_mode'] = selected_game_mode
            return redirect('exploration')

    return render(request, 'select_game_mode.html', {
        'ongoing_quiz': ongoing_quiz,
        'current_category': current_category,
        'current_index': current_index,
        'all_categories_completed': all_categories_completed,
        'completed_categories': completed_categories,
        'all_wave2_questions_answered': all_wave2_questions_answered,

    })


# View function for the about page
def about(request):
    return render(request, 'about.html')

def select_category(request):
    if request.method == 'POST':
        selected_category = request.POST.get('selected_category')
        if selected_category:
            if 'start_time' not in request.session:
                request.session['start_time'] = str(timezone.now())
            
            request.session['current_category'] = selected_category
            request.session['current_index'] = 0

            return redirect('wave1_quiz', category=selected_category)

    completed_categories = request.session.get('completed_categories', [])
    total_categories = len(CATEGORIES)
    all_categories_completed = len(completed_categories) == total_categories
    all_wave2_questions_answered = request.session.get('all_wave2_questions_answered', False)
    total_questions = Wave1Question.objects.count()
    answered_questions = Wave1Question.objects.exclude(user_answer=None).count()
    progress_percentage = (answered_questions / total_questions) * 100 if total_questions > 0 else 0

    # Debug statement for total questions and answered questions
    print(f"Total questions: {total_questions}")
    print(f"Answered questions: {answered_questions}")
    print(f"Progress percentage: {progress_percentage:.2f}%")

    # Get the count of questions per category
    category_question_counts = {}
    for cat in CATEGORIES:
        category_prefix = cat['prefix']
        count = Wave1Question.objects.filter(wave_id__startswith=category_prefix).count()
        category_question_counts[category_prefix] = count

        # Debug statement for each category's question count
        print(f"Category prefix: {category_prefix}, Question count: {count}")

    available_categories = [cat for cat in CATEGORIES if cat['prefix'] not in completed_categories]
    completed_categories_names = [cat['name'] for cat in CATEGORIES if cat['prefix'] in completed_categories]
    
    # Add number of questions to each category
    for cat in available_categories:
        cat['num_questions'] = category_question_counts.get(cat['prefix'], 0)
    
    # Debug statement for available categories with their question counts
    print("Available categories with question counts:")
    for cat in available_categories:
        print(f"Category name: {cat['name']}, Number of questions: {cat['num_questions']}")

    total_available = len(available_categories)
    angle_step = 360 / total_available if total_available > 0 else 0

    for index, category in enumerate(available_categories):
        category['angle'] = angle_step * index

    # Debug statement for category angles
    print("Category angles:")
    for cat in available_categories:
        print(f"Category name: {cat['name']}, Angle: {cat['angle']}")

    return render(request, 'select_category.html', {
        'categories': available_categories,
        'completed_categories': completed_categories_names,
        'all_categories_completed': all_categories_completed,
        'progress_percentage': progress_percentage,
        'all_wave2_questions_answered': all_wave2_questions_answered,
        'category_question_counts': category_question_counts  # Pass the counts to the template
    })


# Check if a category has been completed
def is_category_completed(category, questions):
    return all(question.user_answer is not None for question in questions)

def wave1_quiz(request, category=None):
    # Ensure category is provided and exists
    if not category:
        category = request.session.get('current_category')
        if not category:
            return redirect('select_category')

    # Retrieve or initialize index
    index = request.session.get('current_index', 0)

    # Retrieve questions for the given category
    category_prefix = next((cat['prefix'] for cat in CATEGORIES if cat['prefix'] == category), None)
    if not category_prefix:
        return render(request, 'no_questions.html')
    
    questions = Wave1Question.objects.filter(wave_id__startswith=category_prefix).order_by('id')
    total_questions = questions.count()  # Total number of questions in the category
    if not questions:
        return render(request, 'no_questions.html')

    # Determine the full category name
    full_category_name = next((cat['name'] for cat in CATEGORIES if cat['prefix'] == category_prefix), None)
    
    # Handle POST requests (form submission)
    if request.method == 'POST':
        user_answer = request.POST.get('user_answer', '')
        back_button_clicked = request.POST.get('back_button') == 'true'

        # Handle back button click
        if back_button_clicked and index > 0:
            index -= 1
            request.session['current_index'] = index  # Update session index
            question = questions[index]
            question.user_answer = None
            question.tense = None
            question.save()

        # Handle user answer submission
        elif user_answer in ('yes', 'no'):
            question = questions[index]
            question.user_answer = user_answer == 'yes'
            question.tense = request.POST.get('switch_state') is not None
            question.save()
            index += 1  # Increment to the next question
            request.session['current_index'] = index  # Update session index

        # Check if quiz is completed
        if index >= total_questions:
            if is_category_completed(category, questions):
                completed_categories = set(request.session.get('completed_categories', []))
                if category not in completed_categories:
                    completed_categories.add(category)
                    request.session['completed_categories'] = list(completed_categories)

                all_wave1_questions_answered = Wave1Question.objects.filter(user_answer__isnull=True).count() == 0
                if all_wave1_questions_answered:
                    return redirect('wave2_quiz')
                else:
                    return redirect('quiz_completed', category=category)

        # Redirect to avoid form resubmission on refresh (PRG pattern)
        return HttpResponseRedirect(request.path_info)

    # Handle GET requests (initial page load or after redirection)
    if index < total_questions:
        question = questions[index]
        question.wave1questions = mark_safe(re.sub(r'(<a\b[^>]*href=[\'"][^\'"]*[\'"])([^>]*>)', r'\1 target="_blank"\2', question.wave1questions))
        question.wave1questionspresent = mark_safe(re.sub(r'(<a\b[^>]*href=[\'"][^\'"]*[\'"])([^>]*>)', r'\1 target="_blank"\2', question.wave1questionspresent))
        
        # Render the question page, passing the current index, total questions, and full category name
        return render(request, 'wave1_question.html', {
            'question': question,
            'category_name': category,
            'full_category_name': full_category_name,
            'index': index + 1,  # For 1-based indexing in the template
            'total_questions': total_questions  # Pass total number of questions to the template
        })

    # Redirect to category selection if no more questions
    return redirect('select_category')





# View function for quiz completion
def quiz_completed(request, category):
    # Retrieve completed categories
    completed_categories = request.session.get('completed_categories', [])
    questions = Wave1Question.objects.filter(wave_id__startswith=category, user_answer__isnull=False)

    # Calculate total questions and answered questions
    total_questions = Wave1Question.objects.count()
    answered_questions = Wave1Question.objects.exclude(user_answer=None).count()

    # Calculate progress percentage
    progress_percentage = (answered_questions / total_questions) * 100 if total_questions > 0 else 0

    # Retrieve full category name for display
    category_name = next((cat['name'] for cat in CATEGORIES if cat['prefix'] == category), "Unknown Category")

    return render(request, 'quiz_completed.html', {
        'category_name': category_name,
        'progress_percentage': progress_percentage,
		'questions': questions,
    })
# View function to reset the session
def reset_session(request):
    request.session.flush()
    return redirect('select_category')

# View function to reset user answers
def reset_user_answers(request):
    Wave1Question.objects.update(user_answer=None, tense=None)
    Wave2Question.objects.update(user_answer=None, tense=None)
    EventDetails.objects.update(
        wave2_1_yes_consequence=None,
        wave2_1_no_consequence=None,
        wave2_2_yes_consequence=None,
        wave2_2_no_consequence=None,
        wave2_3_yes_consequence=None,
        wave2_3_no_consequence=None,
        wave2_4_yes_consequence=None,
        wave2_4_no_consequence=None,
        # Add other fields as needed
    )
    return redirect('select_category')

def reset_user_answers_part_2_only(request):
    request.session['answered_wave2_questions'] = 0

    # Reset the relevant database entries for Part 2
    Wave2Question.objects.update(user_answer=None, tense=None)
    EventDetails.objects.update(
        wave2_1_yes_consequence=None,
        wave2_1_no_consequence=None,
        wave2_2_yes_consequence=None,
        wave2_2_no_consequence=None,
        wave2_3_yes_consequence=None,
        wave2_3_no_consequence=None,
        wave2_4_yes_consequence=None,
        wave2_4_no_consequence=None,
    )

    return redirect('select_category') 

# View function to reset session and user answers
def reset_session_and_user_answers(request):
    request.session.flush()
    return reset_user_answers(request)


def wave2_quiz(request):
    wave1_questions = Wave1Question.objects.all()

    context = {
        'questions': wave1_questions
    }
    return render(request, 'wave2_quiz.html', context)

def wave2_bringiton(request):
    # Get all Wave1Question objects where user_answer is True
    wave1_questions = Wave1Question.objects.filter(user_answer=True)
    
    # Get the wave_id values from these Wave1Question objects
    wave1_question_ids = wave1_questions.values_list('wave_id', flat=True)
    print("Wave1 Question IDs:", wave1_question_ids)  # Debug print
    # Construct a query for `wave_id__startswith` for each prefix
    if wave1_question_ids:
        query = Q()
        for wave_id in wave1_question_ids:
            query |= Q(wave_id__startswith=wave_id)
        total_wave2_questions = Wave2Question.objects.filter(query).count()
    else:
        total_wave2_questions = 0    	
    print("Total Wave2 Questions:", total_wave2_questions)  # Debug print
    current_wave1_question = wave1_questions[0]
    wave2_questions = Wave2Question.objects.filter(wave_id__contains=current_wave1_question.wave_id)      
    request.session['total_wave2_questions'] = total_wave2_questions
    request.session['answered_wave2_questions'] = 0  # Initialize answered question count
    wave2_question_ids = [wave2_question.id for wave2_question in wave2_questions]
    # Assuming there is a default Wave2Question when the page loads
    current_wave2_question = wave2_questions.first()

    event_details = EventDetails.objects.get(wave_id=current_wave2_question.wave_id)

    context = {
        'wave1_questions': wave1_questions,
        'wave2_questions': wave2_questions,
        'wave2_question_ids': wave2_question_ids,  # Convert queryset to a list
        'current_wave1_question': current_wave1_question,
        'current_wave2_question': current_wave2_question,  # Pass the initial wave2 question
        'event_details': event_details,
		'total_wave2_questions': total_wave2_questions,
    }

    if request.method == 'POST':
        wave2_question_id = request.POST.get('wave2-question-id')
        user_answer = request.POST.get('wave2-yes') or request.POST.get('wave2-no')
        tense_value = request.POST.get('switch_state')

        # Get the Wave2Question object for the submitted answer
        wave2_question = get_object_or_404(Wave2Question, id=wave2_question_id)


        wave2_question.tense = True if tense_value == '1' else False

        # Save the user's answer to the corresponding Wave2Question
        wave2_question.user_answer = user_answer
        wave2_question.save()

        # Optionally, you might want to get the next Wave2Question and update the context

    return render(request, 'wave2_bringiton.html', context)

def wave2_bringiton_data(request):
    wave1_questions = Wave1Question.objects.filter(user_answer=True)

    if not wave1_questions:
        return JsonResponse({'error': 'No user-answered wave1 questions'})

    all_wave2_questions_answered = all(is_wave2_questions_answered(wave1_question) for wave1_question in wave1_questions)

    if all_wave2_questions_answered:
        # Redirect to quiz_completion if all Wave2Questions for affirmatively answered Wave1Questions are answered
        request.session['all_wave2_questions_answered'] = True
        return render(request, 'quiz_completion.html')

    current_wave1_question_index = request.session.get('current_wave1_question_index', 0)
    current_wave1_question = wave1_questions[current_wave1_question_index]

    # Retrieve the wave2 questions related to the current wave1 question
    wave2_questions = Wave2Question.objects.filter(wave_id__contains=current_wave1_question.wave_id)

    if not wave2_questions:
        return JsonResponse({'error': 'No wave2 questions found for the current wave1 question'})

    current_wave2_question_index = request.session.get('current_wave2_question_index', 0)
    current_wave2_question = None

    if current_wave2_question_index < len(wave2_questions):
        current_wave2_question = wave2_questions[current_wave2_question_index]

    if current_wave2_question is not None:
        event_details = EventDetails.objects.get(wave_id=current_wave2_question.wave_id)

        data = {
            'wave2_question_id': current_wave2_question.id,  # Include wave2_question_id in the response
            'wave2_question': current_wave2_question.wave2_question,
            'wave2_question_present': current_wave2_question.wave2_question_present,
            'wave1_tense': current_wave1_question.tense,
            'wave2_wave_id': current_wave2_question.wave_id,  # Include wave2 wave_id
            'wave1_wave_id': current_wave1_question.wave_id,  # Include wave1 wave_id
            'wave1_question': {
                'wave1questions': current_wave1_question.wave1questions,
                'wave1questionspresent': current_wave1_question.wave1questionspresent,
            },
            'event_details': {
                'wave2_1_no_details': event_details.wave2_1_no_details,
                'wave2_1_yes_details': event_details.wave2_1_yes_details,
                'wave2_2_no_details': event_details.wave2_2_no_details,
                'wave2_2_yes_details': event_details.wave2_2_yes_details,
                'wave2_3_no_details': event_details.wave2_3_no_details,
                'wave2_3_yes_details': event_details.wave2_3_yes_details,
                'wave2_4_no_details': event_details.wave2_4_no_details,
                'wave2_4_yes_details': event_details.wave2_4_yes_details,
                'wave2_1_yes_consequence': event_details.wave2_1_yes_consequence,
                'wave2_1_no_consequence': event_details.wave2_1_no_consequence,
                'wave2_2_yes_consequence': event_details.wave2_2_yes_consequence,
                'wave2_2_no_consequence': event_details.wave2_2_no_consequence,
                'wave2_3_yes_consequence': event_details.wave2_3_yes_consequence,
                'wave2_3_no_consequence': event_details.wave2_3_no_consequence,
                'wave2_4_yes_consequence': event_details.wave2_4_yes_consequence,
                'wave2_4_no_consequence': event_details.wave2_4_no_consequence,
                # Add other fields here as needed
            },              # Include the tense value
        }

        # Update the current wave2 question index in the session
        request.session['current_wave2_question_index'] = current_wave2_question_index + 1

        return JsonResponse(data)
    else:
        # If there are no more wave2 questions for the current wave1 question, proceed to the next wave1 question
        current_wave1_question_index = (current_wave1_question_index + 1) % len(wave1_questions)
        request.session['current_wave1_question_index'] = current_wave1_question_index

        # Get the new wave1 question
        current_wave1_question = wave1_questions[current_wave1_question_index]

        # Retrieve the wave2 questions related to the new wave1 question
        wave2_questions = Wave2Question.objects.filter(wave_id__contains=current_wave1_question.wave_id)

        if wave2_questions:
            # Get the first wave2 question for the new wave1 question
            current_wave2_question = wave2_questions[0]

            # Fetch the EventDetails object for the new wave2_question
            event_details = EventDetails.objects.get(wave_id=current_wave2_question.wave_id)

            # Create a dictionary containing the data you want to send back
            data = {
                'wave2_question_id': current_wave2_question.id,  # Include wave2_question_id in the response
                'wave2_question': current_wave2_question.wave2_question,
                'wave2_question_present': current_wave2_question.wave2_question_present,
                'wave1_tense': current_wave1_question.tense,  # Include the tense value
                'wave2_wave_id': current_wave2_question.wave_id,  # Include wave2 wave_id
                'wave1_wave_id': current_wave1_question.wave_id,  # Include wave1 wave_id
                'wave1_question': {
                    'wave1questions': current_wave1_question.wave1questions,
                    'wave1questionspresent': current_wave1_question.wave1questionspresent,
                },
                'event_details': {
                    'wave2_1_no_details': event_details.wave2_1_no_details,
                    'wave2_1_yes_details': event_details.wave2_1_yes_details,
                    'wave2_2_no_details': event_details.wave2_2_no_details,
                    'wave2_2_yes_details': event_details.wave2_2_yes_details,
                    'wave2_3_no_details': event_details.wave2_3_no_details,
                    'wave2_3_yes_details': event_details.wave2_3_yes_details,
                    'wave2_4_no_details': event_details.wave2_4_no_details,
                    'wave2_4_yes_details': event_details.wave2_4_yes_details,
                    'wave2_1_yes_consequence': event_details.wave2_1_yes_consequence,
                    'wave2_1_no_consequence': event_details.wave2_1_no_consequence,
                    'wave2_2_yes_consequence': event_details.wave2_2_yes_consequence,
                    'wave2_2_no_consequence': event_details.wave2_2_no_consequence,
                    'wave2_3_yes_consequence': event_details.wave2_3_yes_consequence,
                    'wave2_3_no_consequence': event_details.wave2_3_no_consequence,
                    'wave2_4_yes_consequence': event_details.wave2_4_yes_consequence,
                    'wave2_4_no_consequence': event_details.wave2_4_no_consequence,
                    # Add other fields here as needed
                }

            }

            # Update the current wave2 question index in the session
            request.session['current_wave2_question_index'] = 1


            return JsonResponse(data)
        else:
            # No more wave2 questions for the new wave1 question, proceed accordingly
            return render(request, 'quiz_completion.html')


def quiz_completion(request):
    # Calculate the time taken
    start_time_str = request.session.get('start_time')
    if start_time_str:
        start_time = timezone.datetime.fromisoformat(start_time_str)
        end_time = timezone.now()
        time_taken = end_time - start_time
        
        # Calculate hours, minutes, and seconds
        total_seconds = int(time_taken.total_seconds())
        hours, remainder = divmod(total_seconds, 3600)
        minutes, seconds = divmod(remainder, 60)
        
        # Format time as hh:mm:ss
        formatted_time = f"{hours:02}:{minutes:02}:{seconds:02}"
        
        # Remove start_time from session as it's no longer needed
        del request.session['start_time']
    else:
        formatted_time = "00:00:00"  # Default value if no start time found

    # Fetch all wave1questions from the database
    wave1_questions = Wave1Question.objects.all()

    # Separate questions based on user_answer
    wave1_questions_affirmative = wave1_questions.filter(user_answer=True)
    wave1_questions_negative = wave1_questions.filter(user_answer=False)

    wave1_questions_data = []

    # Iterate over affirmative wave1_questions
    for wave1_question in wave1_questions_affirmative:
        # Retrieve related wave2_questions for the current wave1_question
        related_wave2_questions = Wave2Question.objects.filter(wave_id__startswith=wave1_question.wave_id)

        # Retrieve related EventDetails for the current wave1_question
        related_event_details = EventDetails.objects.filter(wave_id__in=related_wave2_questions.values_list('wave_id', flat=True))

        # Separate related_event_details for yes and no
        related_event_details_yes = related_event_details.filter(wave2_1_yes_details__isnull=False)
        related_event_details_no = related_event_details.filter(wave2_1_no_details__isnull=False)

        wave2_2_yes_details = related_event_details.values_list('wave2_2_yes_details', flat=True).first()
        wave2_2_no_details = related_event_details.values_list('wave2_2_no_details', flat=True).first()
        wave2_3_yes_details = related_event_details.values_list('wave2_3_yes_details', flat=True).first()
        wave2_3_no_details = related_event_details.values_list('wave2_3_no_details', flat=True).first()
        wave2_4_yes_details = related_event_details.values_list('wave2_4_yes_details', flat=True).first()
        wave2_4_no_details = related_event_details.values_list('wave2_4_no_details', flat=True).first()
        wave2_1_yes_consequence = related_event_details.values_list('wave2_1_yes_consequence', flat=True).first()
        wave2_1_no_consequence = related_event_details.values_list('wave2_1_no_consequence', flat=True).first()
        wave2_2_yes_consequence = related_event_details.values_list('wave2_2_yes_consequence', flat=True).first()
        wave2_2_no_consequence = related_event_details.values_list('wave2_2_no_consequence', flat=True).first()
        wave2_3_yes_consequence = related_event_details.values_list('wave2_3_yes_consequence', flat=True).first()
        wave2_3_no_consequence = related_event_details.values_list('wave2_3_no_consequence', flat=True).first()
        wave2_4_yes_consequence = related_event_details.values_list('wave2_4_yes_consequence', flat=True).first()
        wave2_4_no_consequence = related_event_details.values_list('wave2_4_no_consequence', flat=True).first()

        wave1_question_data = {
            'wave1_question': wave1_question,
            'wave1_wave_id': wave1_question.wave_id,
            'related_wave2_questions': related_wave2_questions,
            'related_event_details_yes': related_event_details_yes,
            'related_event_details_no': related_event_details_no,
            'wave2_1_yes_consequence': wave2_1_yes_consequence,
            'wave2_1_no_consequence': wave2_1_no_consequence,
            'wave2_2_yes_consequence': wave2_2_yes_consequence,
            'wave2_2_no_consequence': wave2_2_no_consequence,
            'wave2_3_yes_consequence': wave2_3_yes_consequence,
            'wave2_3_no_consequence': wave2_3_no_consequence,
            'wave2_4_yes_consequence': wave2_4_yes_consequence,
            'wave2_4_no_consequence': wave2_4_no_consequence,
            'wave2_2_yes_details': wave2_2_yes_details,
            'wave2_2_no_details': wave2_2_no_details,
            'wave2_3_yes_details': wave2_3_yes_details,
            'wave2_3_no_details': wave2_3_no_details,
            'wave2_4_yes_details': wave2_4_yes_details,
            'wave2_4_no_details': wave2_4_no_details,

        }

        wave1_questions_data.append(wave1_question_data)

    context = {
        'wave1_questions_data': wave1_questions_data,
        'wave1_questions_negative': wave1_questions_negative,
		'time_taken': formatted_time
    }

    return render(request, 'quiz_completion.html', context)


def is_wave2_questions_answered(wave1_question):
    related_wave2_questions = Wave2Question.objects.filter(wave_id__startswith=wave1_question.wave_id)
    for wave2_question in related_wave2_questions:
        event_details = EventDetails.objects.filter(wave_id=wave2_question.wave_id).first()
        if not event_details:
            return False

        for current_order in range(1, 5):
            yes_consequence_value = getattr(event_details, f"wave2_{current_order}_yes_consequence")
            no_consequence_value = getattr(event_details, f"wave2_{current_order}_no_consequence")

            if yes_consequence_value == 3 or no_consequence_value == 3:
                break  # At least one wave2_question is answered, move to the next

        else:
            return False  # None of the consequences were 3 for this wave2_question

    return True  # All wave2_questions for this wave1_question are answered

@require_POST
def update_user_answer(request):
    data = json.loads(request.body)
    wave2_question_id = data.get('wave2_question_id')
    user_answer = data.get('user_answer')
    tense_value = data.get('tense')  # Retrieve the 'tense' value from the data

    try:
        # Check if user_answer and tense are both None
        if user_answer is None and tense_value is None:
            # If both are None, update the specific Wave2Question object to set both user_answer and tense as null
            Wave2Question.objects.filter(pk=wave2_question_id).update(user_answer=None, tense=None)
            return JsonResponse({'success': True})
        # Update the specific Wave2Question object using the "id" field
        question = Wave2Question.objects.get(pk=wave2_question_id)
        question.user_answer = user_answer
        question.tense = tense_value  # Set the 'tense' field
        question.save()

        return JsonResponse({'success': True})
    except Wave2Question.DoesNotExist:
        return JsonResponse({'success': False, 'error': 'Question not found'})
    except Exception as e:
        return JsonResponse({'success': False, 'error': str(e)})


def handle_event_details_consequences(request):
    if request.method == 'POST':
        try:
            data = json.loads(request.body)
            print("Received data:", data)

            wave2_question_id = data.get('wave2_question_id')
            target_order = data.get('target_order')
            consequence_value = data.get('consequence_value')

            print("Wave2 Question ID:", wave2_question_id)
            print("Target Order:", target_order)
            print("Consequence Value:", consequence_value)

            if consequence_value == 'clear':
                # Clear the values in the database for both "yes" and "no" columns
                EventDetails.objects.filter(pk=wave2_question_id).update(**{
                    f"wave2_{target_order}_yes_consequence": None,
                    f"wave2_{target_order}_no_consequence": None,
                })
                print("Cleared event details for Wave2 question ID:", wave2_question_id)
                return JsonResponse({'success': True})

            # Get the corresponding Wave2Question object
            wave2_question = Wave2Question.objects.get(pk=wave2_question_id)

            print("Wave2 Question:", wave2_question)

            # Now you can use the wave_id from the Wave2Question object
            event_detail = EventDetails.objects.get(wave_id=wave2_question.wave_id)

            print("Event Details:", event_detail)

            # Construct the field name based on target_order and consequence_value
            if target_order == 1:
                field_name = f"wave2_1_{('yes' if wave2_question.user_answer == 1 else 'no')}_consequence"
                # Set the value for the dynamically constructed fields
                setattr(event_detail, field_name, consequence_value)  # Assuming 1 needs to be saved
                print(f"Set {field_name} to {consequence_value}")
                event_detail.save()
            else:
                prev_field_name = f"wave2_{target_order - 1}_{('yes' if wave2_question.user_answer == 1 else 'no')}_consequence"
                prev_value = getattr(event_detail, prev_field_name, None)

                next_field_name = f"wave2_{target_order}_{('yes' if prev_value == 1 else 'no')}_consequence"
                setattr(event_detail, next_field_name, consequence_value)  # Assuming 1 needs to be saved
                print(f"Set {next_field_name} to {consequence_value}")
                event_detail.save()

            print("Updated event details successfully")
            return JsonResponse({'success': True})
        except Exception as e:
            # Handle exceptions appropriately
            print(f"Error handling event details consequences: {e}")

    return JsonResponse({'success': False})

def exploration(request):
    # Calculate the total number of exploration data items
    exploration_total = ExplorationData.objects.all().count()

    # Retrieve the index representing the number of unique items accessed
    accessed_index = request.session.get('accessed_index', 0)
    accessed_index = int(accessed_index)  # Ensure it's an integer

    # Calculate the overall progress percentage across all categories
    progress_percentage = int((accessed_index / exploration_total) * 100) if exploration_total > 0 else 0

    return render(request, 'exploration.html', {'progress_percentage': progress_percentage})

def exploration_search(request):
    # Retrieve filter parameters
    rdlc = request.GET.get('rdlc', '')
    harmful_effects = request.GET.get('harmful_effects', '')
    possible_causes = request.GET.get('possible_causes', '')
    search_query = request.GET.get('search_query', '')  # Custom search query

    # Retrieve sorting parameter
    sort_by = request.GET.get('sort_by', 'alphabetical_asc')  # Default to alphabetical ascending

    # Prepare filters using `icontains` for substring matching
    filters = {}
    if rdlc:
        filters['rdlc__icontains'] = rdlc
    if harmful_effects:
        filters['harmful_effects__icontains'] = harmful_effects
    if possible_causes:
        filters['possible_causes__icontains'] = possible_causes
    if search_query:
        filters['topic__icontains'] = search_query  # Case-insensitive search by topic name

    # Fetch data and apply filters
    results = ExplorationData.objects.filter(**filters)

    # Apply sorting based on the selected option
    if sort_by == 'alphabetical_asc':
        results = results.order_by('topic')
    elif sort_by == 'alphabetical_desc':
        results = results.order_by('-topic')

    # Get the count of results
    results_count = results.count()

    return render(request, 'exploration_search_results.html', {
        'results': results,
        'results_count': results_count,
        'rdlc': rdlc,
        'harmful_effects': harmful_effects,
        'possible_causes': possible_causes,
        'search_query': search_query,
        'sort_by': sort_by,
    })




def exploration_category(request, category):
    sort_by = request.GET.get('sort_by', 'alphabetical_asc')

    # Retrieve the exploration data filtered by category
    exploration_data_list = ExplorationData.objects.filter(rdlc=category)

    # Apply sorting based on the 'sort_by' parameter
    if sort_by == 'alphabetical_asc':
        exploration_data_list = exploration_data_list.order_by('topic')
    elif sort_by == 'alphabetical_desc':
        exploration_data_list = exploration_data_list.order_by('-topic')
    elif sort_by == 'latest':
        exploration_data_list = exploration_data_list.order_by('-created_at')
    elif sort_by == 'oldest':
        exploration_data_list = exploration_data_list.order_by('created_at')

    # Calculate the total number of exploration data items across all categories
    exploration_total = ExplorationData.objects.all().count()

    # Retrieve the index representing the number of unique items accessed
    accessed_index = request.session.get('accessed_index', 0)
    accessed_ids = request.session.get('accessed_ids', [])

    # Calculate the progress percentage for category-wise discovery
    category_items = exploration_data_list.count()
    category_progress_percentage = int((accessed_index / category_items) * 100) if category_items > 0 else 0

    # Calculate the overall progress percentage across all categories
    progress_percentage = int((accessed_index / exploration_total) * 100) if exploration_total > 0 else 0

    # Pass the sorting option to the template
    context = {
        'category': category,
        'exploration_data_list': exploration_data_list,
        'progress_percentage': progress_percentage,
        'category_progress_percentage': category_progress_percentage,
        'accessed_ids': accessed_ids,
        'sort_by': sort_by  # Pass the current sort option to the template
    }
    
    return render(request, 'exploration_category.html', context)

def exploration_detail(request, id):
    # Retrieve the exploration object based on the exploration_id
    exploration = ExplorationData.objects.get(pk=id)

    # Assuming exploration.rdlc is a field in the Exploration model
    category = exploration.rdlc

    # Retrieve the index representing the number of unique items accessed
    accessed_index = request.session.get('accessed_index', 0)

    # Check if the ID is already accessed
    accessed_ids = request.session.get('accessed_ids', [])
    if id not in accessed_ids:
        # Increment the index and store it in the session along with the ID
        accessed_index += 1
        accessed_ids.append(id)
        request.session['accessed_index'] = accessed_index
        request.session['accessed_ids'] = accessed_ids

    # Calculate the total number of exploration data items across all categories
    exploration_total = ExplorationData.objects.all().count()

    # Recalculate overall progress percentage
    progress_percentage = int((accessed_index / exploration_total) * 100) if exploration_total > 0 else 0

    # Recalculate category-wise progress percentage
    exploration_data_list = ExplorationData.objects.filter(rdlc=category)
    category_items = exploration_data_list.count()
    category_progress_percentage = int((accessed_index / category_items) * 100) if category_items > 0 else 0

    context = {
        'exploration': exploration,
        'category': category,
        'progress_percentage': progress_percentage,
        'category_progress_percentage': category_progress_percentage,
    }

    return render(request, 'exploration_detail.html', context)


def reset_progress(request):
    # Reset accessed_index in session to 0
    del request.session['accessed_index']
    del request.session['accessed_ids']
    return redirect('exploration')
