import sqlite3
import re
from bs4 import BeautifulSoup

# Path to your database file
DATABASE_FILE = 'db.sqlite3'

# Path to your Twine HTML file
TWINE_FILE = 'CRAMP Trees.html'

def parse_twine_file(file_path):
    """Parse the Twine HTML file and extract passages."""
    with open(file_path, 'r', encoding='utf-8') as file:
        soup = BeautifulSoup(file, 'html.parser')
        passages = soup.find_all('tw-passagedata')
        return {passage['name']: clean_passage_text(passage.text) for passage in passages}

def clean_passage_text(text):
    """Remove Twine-specific link statements from the passage content."""
    # Remove Twine link statements in the format [[Link Text | Target]]
    return re.sub(r'\[\[.*?\|.*?\]\]', '', text).strip()

def update_wave1question(passage_name, passage_content):
    """Update the wave1question table in the database."""
    cursor.execute(
        "UPDATE appquestgame_wave1question SET wave1questions = ? WHERE wave_id = ?",
        (passage_content, passage_name)
    )

def update_wave2question(passage_name, passage_content):
    """Update the wave2question table in the database."""
    cursor.execute(
        "UPDATE appquestgame_wave2question SET wave2_question = ? WHERE wave_id = ?",
        (passage_content, passage_name)
    )

def update_eventdetails(passage_name, passage_content):
    """Update the eventdetails table in the database."""
    # Determine the specific field to update based on passage_name
    wave_id, detail_type = passage_name.split('_wave2_', 1)
    field = f'wave2_{detail_type}'
    
    cursor.execute(
        f"UPDATE appquestgame_eventdetails SET {field} = ? WHERE wave_id = ?",
        (passage_content, wave_id)
    )

def update_database(passages, database_file):
    """Update the database based on parsed passages."""
    global cursor
    conn = sqlite3.connect(database_file)
    cursor = conn.cursor()

    for passage_name, passage_content in passages.items():
        if passage_name.startswith("P") or passage_name.startswith("R"):
            if '_wave2_' in passage_name:
                update_eventdetails(passage_name, passage_content)
            elif '_question' in passage_name:
                update_wave2question(passage_name.replace('_question', ''), passage_content)
            else:
                update_wave1question(passage_name, passage_content)
    
    conn.commit()
    conn.close()

# Main script execution
if __name__ == '__main__':
    passages = parse_twine_file(TWINE_FILE)
    update_database(passages, DATABASE_FILE)
    print("Database updated successfully!")
